<?php

use Smartsupp\Auth\Api;


/**
 * Smartsupp Live Chat.
 *
 * @package   Smartsupp_Admin
 * @author    Smartsupp <vladimir@smartsupp.com>
 * @license   GPL-2.0+
 * @link      http://www.smartsupp.com
 * @copyright 2014 smartsupp.com
 */
class Smartsupp_Admin {


	const OPTION_NAME = 'smartsupp';

	/**
	 * @var $this
	 */
	protected static $instance = null;

    /**
     * @var string
     */
    protected $plugin_slug = '';


	private function __construct() {
		$plugin            = Smartsupp::get_instance();
		$this->plugin_slug = $plugin->get_plugin_slug();

		add_action( 'admin_menu', array( $this, 'addMenuItems' ) );
		add_action( 'admin_init', array( $this, 'performAction' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueueAssets' ) );

		$plugin_basename = plugin_basename( plugin_dir_path( realpath( __DIR__ ) ) . $this->plugin_slug . '.php' );
		add_filter(
			'plugin_action_links_' . $plugin_basename,
			function ( $links ) use ( $plugin ) {
				array_unshift( $links, '<a href="options-general.php?page=' . esc_attr( $plugin->get_plugin_slug() ) . '">Settings</a>' );
				return $links;
			}
		);
	}

	public static function get_instance() {
		if ( null == self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}


	public static function install() {
		global $wp_version;

		$checks = array(
			'Your Wordpress version is not compatible with Smartsupp plugin which requires at least version 3.0. Please update your Wordpress or insert Smartsupp chat code into your website manually (you will find the chat code in the email we have sent you upon registration)' => version_compare( $wp_version, '3.0', '<' ),
			'This plugin requires at least PHP version 5.3.2, your version: ' . PHP_VERSION . '. Please ask your hosting company to bring your PHP version up to date.' => version_compare( PHP_VERSION, '5.3.2', '<' ),
			'This plugin requires PHP extension \'curl\' installed and enbaled.' => ! function_exists( 'curl_init' ),
			'This plugin requires PHP extension \'json\' installed and enbaled.' => ! function_exists( 'json_decode' ),
		);

		foreach ( $checks as $message => $disable ) {
			if ( $disable ) {
				deactivate_plugins( basename( __FILE__ ) );
				wp_die( esc_html( $message ) );
			}
		}

		update_option(
			self::OPTION_NAME,
			array(
				'active'        => false,
				'chat-id'       => null,
				'email'         => null,
				'optional-code' => null,
			)
		);
	}


	public function addMenuItems() {
		add_options_page(
			__( 'Smartsupp Live Chat - Settings', 'smartsupp-live-chat' ),
			__( 'Smartsupp Live Chat', 'smartsupp-live-chat' ),
			'manage_options',
			$this->plugin_slug,
			array( $this, 'renderAdminPage' )
		);

		add_menu_page(
			__( 'Smartsupp Live Chat - Settings', 'smartsupp-live-chat' ),
			__( 'Smartsupp', 'smartsupp-live-chat' ),
			'manage_options',
			$this->plugin_slug,
			array( $this, 'renderAdminPage' ),
			plugins_url( 'images/icon-20x20.png', __DIR__ )
		);
	}


	public function performAction() {
		if ( ! is_user_logged_in() ) {
			return;
		}
		if ( ! current_user_can('manage_options') ) {
			return;
		}
		if ( ! isset( $_GET['ssaction'] ) ) {
			return;
		}

		$formAction = $message = $email = $marketingConsent = $termsConsent = null;
		$action     = sanitize_text_field( wp_unslash( $_GET['ssaction'] ) );
		switch ( $action ) {
			case 'login':
			case 'register':
                if ( ! isset( $_POST['email'] ) || ! isset( $_POST['password'] ) ) {
                    return;
                }
				if ( ! isset( $_POST['_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_nonce'] ) ), 'smartsupp' ) ) {
					$message = 'Invalid nonce';
					break;
				}
				$formAction = $action;
				$api        = new Api();
				$data       = array(
					'email'            => sanitize_email( wp_unslash( $_POST['email'] ) ),
					'password'         => sanitize_text_field( wp_unslash( $_POST['password'] ) ),
					'consentTerms'     => 1,
					'consentMarketing' => 1,
					'platform'         => 'WP ' . get_bloginfo( 'version' ),
				);
				try {
					$response = $formAction === 'login' ? $api->login( $data ) : $api->create( $data + array( 'partnerKey' => 'k717wrqdi5' ) );

					if ( isset( $response['error'] ) ) {
						$message = $response['message'];
						$email   = sanitize_email( wp_unslash( $_POST['email'] ) );
					} else {
						$this->activate( $response['account']['key'], sanitize_email( wp_unslash( $_POST['email'] ) ) );
					}
				} catch ( Exception $e ) {
					$message          = $e->getMessage();
					$email            = sanitize_email( wp_unslash( $_POST['email'] ) );
					$marketingConsent = 1;
					$termsConsent     = 1;
				}
				break;
			case 'update':
                if ( ! isset( $_POST['code'] ) ) {
                    return;
                }
                if ( ! current_user_can('unfiltered_html') ) {
                    $message = 'Not allowed';
                    break;
                }
				if ( ! isset( $_POST['_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_nonce'] ) ), 'smartsupp_update' ) ) {
					$message = 'Invalid nonce';
					break;
				}
				$message = 'Custom code was updated.';
				$this->updateOptions(
					array(
						'optional-code' => sanitize_textarea_field( wp_unslash( $_POST['code'] ) ),
					)
				);
				break;
			case 'disable':
				if ( ! isset( $_POST['_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_nonce'] ) ), 'smartsupp_disable' ) ) {
					$message = 'Invalid nonce';
					break;
				}
				$this->deactivate();
				break;
			default:
				$message = 'Invalid action';
				break;
		}
		$this->renderAdminPage( $message, $formAction, $email, $marketingConsent, $termsConsent );
		exit;
	}


	public function renderAdminPage( $message = null, $formAction = null, $email = null, $marketingConsent = null, $termsConsent = null ) {
		$this->render(
			'views/admin.php',
			array(
				'domain'           => $this->plugin_slug,
				'customers'        => $this->getCustomers(),
				'features'         => $this->getFeatures(),
				'options'          => $this->getOptions(),
				'message'          => (string) $message,
				'formAction'       => $formAction,
				'email'            => $email,
				'marketingConsent' => $marketingConsent,
				'termsConsent'     => $termsConsent,
			)
		);
	}


	private function render( $template, $vars = array() ) {
		call_user_func_array(
			function () use ( $template, $vars ) {
				extract( $vars );
                // nosemgrep audit.php.lang.security.file.inclusion-arg.
				include $template;
			},
			array()
		);
	}

	private function getFeatures() {
		$pluginUrl = $this->getPluginUrl();
		return '<section class="features">' .
			'<div class="features__container">' .
				'<div class="features__item">' .
					'<span class="features__header">' .
						esc_html__( 'MULTICHANNEL', 'smartsupp-live-chat' ) .
					'</span>' .
					'<img src="' . esc_url( $pluginUrl ) . '/images/multichannel-fb.png">' .
					'<h2 class="features__item-title">' .
						esc_html__( 'Respond to customers\' chats and emails from one place', 'smartsupp-live-chat' ) .
					'</h2>' .
				'</div>' .
				'<div class="features__item">' .
					'<span class="features__header">' .
						esc_html__( 'CHAT BOT', 'smartsupp-live-chat' ) .
					'</span>' .
					'<img src="' . esc_url( $pluginUrl ) . '/images/chatbot.png">' .
					'<h2 class="features__item-title">' .
						esc_html__( 'Engage your visitors with automated chat bot', 'smartsupp-live-chat' ) .
					'</h2>' .
				'</div>' .
				'<div class="features__item">' .
					'<span class="features__header">' .
						esc_html__( 'MOBILE APP', 'smartsupp-live-chat' ) .
					'</span>' .
					'<img src="' . esc_url( $pluginUrl ) . '/images/mobile.png">' .
					'<h2 class="features__item-title">' .
						esc_html__( 'Chat with customers on the go with app for iOS & Android', 'smartsupp-live-chat' ) .
					'</h2>' .
				'</div>' .
			'</div>' .
			'<div class="features__all">' .
				'<a href="https://smartsupp.com" target="_blank" class="btn btn--link btn--arrow">' .
					esc_html__( 'Explore All Features on our website', 'smartsupp-live-chat' ) .
				'</a>' .
			'</div>' .
		'</section>';
	}

	private function getCustomers() {
		$pluginUrl = $this->getPluginUrl();
		return '<section class="clients">' .
			'<div class="clients__container">' .
				'<div class="clients__pretitle">' .
					esc_html__( 'POPULAR CHAT SOLUTION OF EUROPEAN WEBSHOPS AND WEBSITES', 'smartsupp-live-chat' ) .
				'</div>' .
				'<h2 class="clients__title">' .
					esc_html__( 'Join the 338 445 companies and freelancers relying on Smartsupp', 'smartsupp-live-chat' ) .
				'</h2>' .
				'<div class="clients__logos">' .
					'<img src="' . esc_url( $pluginUrl ) . '/images/insportline.png" alt="insportline" />' .
					'<img src="' . esc_url( $pluginUrl ) . '/images/redfox.png" alt="redfox" />' .
					'<img src="' . esc_url( $pluginUrl ) . '/images/motorgarten.png" alt="motorgarten" />' .
					'<img src="' . esc_url( $pluginUrl ) . '/images/travelking.png" alt="travelking" />' .
			'</div>' .
		'</section>';
	}


	private function activate( $chatId, $email ) {
		$this->updateOptions(
			array(
				'active'  => true,
				'chat-id' => (string) $chatId,
				'email'   => (string) $email,
			)
		);
	}


	private function deactivate() {
		$this->updateOptions(
			array(
				'active'  => false,
				'chat-id' => null,
				'email'   => null,
			)
		);
	}


	private function updateOptions( array $options ) {
		$current = $this->getOptions();
		foreach ( $options as $key => $option ) {
			$current[ $key ] = $option;
		}
		update_option( self::OPTION_NAME, $current );
	}


	/**
	 * @return array
	 */
	private function getOptions() {
		return get_option( self::OPTION_NAME );
	}


	/**
	 * @return string
	 */
	private function getPluginUrl() {
		return plugins_url( '', __DIR__ );
	}


	/**
	 * @internal This method is intended just for static analysis of translations
	 */
	private function getMessages() {
		return array(
			__( 'Email does not exist', 'smartsupp-live-chat' ),
			__( 'Invalid password', 'smartsupp-live-chat' ),
			__( 'Email is required', 'smartsupp-live-chat' ),
			__( 'Email already exists', 'smartsupp-live-chat' ),
			__( 'Password is too short. Minimal length is 6 characters.', 'smartsupp-live-chat' ),
			__( 'Custom code was updated.', 'smartsupp-live-chat' ),
			__( 'Invalid action', 'smartsupp-live-chat' ),
		);
	}

	public function enqueueAssets() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( empty( $_GET['page'] ) || $_GET['page'] !== $this->plugin_slug ) {
			return;
		}
		wp_enqueue_style( 'smartsupp-admin-fonts', 'https://fonts.googleapis.com/css?family=Poppins:400,500,600,700&display=swap', array(), null );
		wp_enqueue_style( 'smartsupp-admin', esc_url( plugins_url( 'assets/style.css', dirname( __FILE__ ) ) ), array(), null );
	}
}
