<?php
/**
 * Elementor Animated Heading
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

use Elementor\Group_Control_Css_Filter;
use Elementor\Repeater;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;
use Elementor\Utils;


defined( 'ABSPATH' ) || die();

class Rsaddon_Elementor_pro_Animated_Heading_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve rsgallery widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'rs-animated-heading';
	}		

	/**
	 * Get widget title.
	 *
	 * Retrieve rsgallery widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'RS Animated Heading', 'rsaddon' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve rsgallery widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'rs-badge';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the rsgallery widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
        return [ 'rsaddon_category' ];
    }

	/**
	 * Register rsgallery widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'global_content_section',
			[
				'label' => esc_html__( 'Global', 'rsaddon' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_responsive_control(
            'align',
            [
                'label' => esc_html__( 'Alignment', 'rsaddon' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
						'title' => esc_html__( 'Left', 'rsaddon' ),
						'icon'  => 'eicon-text-align-left',
                    ],
                    'center' => [
						'title' => esc_html__( 'Center', 'rsaddon' ),
						'icon'  => 'eicon-text-align-center',
                    ],
                    'right' => [
						'title' => esc_html__( 'Right', 'rsaddon' ),
						'icon'  => 'eicon-text-align-right',
                    ],
                    'justify' => [
						'title' => esc_html__( 'Justify', 'rsaddon' ),
						'icon'  => 'eicon-text-align-justify',
                    ],
                ],
                'toggle' => true,
                'selectors' => [
                    '{{WRAPPER}} .rs-animated-heading' => 'text-align: {{VALUE}}'
                ]
            ]
        );
		$this->end_controls_section();


		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Normal Title', 'rsaddon' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);


		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Normal Text', 'rsaddon' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Animated Heading', 'rsaddon' ),				
				'separator' => 'before',
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'   => esc_html__( 'Select Heading Tag', 'rsaddon' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'h2',
				'options' => [						
					'h1' => esc_html__( 'H1', 'rsaddon'),
					'h2' => esc_html__( 'H2', 'rsaddon'),
					'h3' => esc_html__( 'H3', 'rsaddon' ),
					'h4' => esc_html__( 'H4', 'rsaddon' ),
					'h5' => esc_html__( 'H5', 'rsaddon' ),
					'h6' => esc_html__( 'H6', 'rsaddon' ),
					'div' => esc_html__( 'div', 'rsaddon' ),
				],
			]
		);
		$this->end_controls_section();


		$this->start_controls_section(
			'animated_section',
			[
				'label' => esc_html__( 'Animated Title', 'rsaddon' ),
				'tab' => Controls_Manager::TAB_CONTENT,							
			]
		);
		$this->add_control(
			'animated_text',
			[
				'label' => esc_html__( 'Animate Content', 'rsaddon' ),
				'type' => Controls_Manager::TEXTAREA,	
				'description' => esc_html__('Use double comma (e.g. First Line,,).', 'rsaddon'),
			]
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'content_section_after',
			[
				'label' => esc_html__( 'After Animate Title', 'rsaddon' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);


		$this->add_control(
			'title_after',
			[
				'label' => esc_html__( 'Text', 'rsaddon' ),
				'type' => Controls_Manager::TEXT,							
				'separator' => 'before',
			]
		);
	
		$this->end_controls_section();


		$this->start_controls_section(
			'des_section',
			[
				'label' => esc_html__( 'Description Text', 'rsaddon' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'content',
			[
				'type' => Controls_Manager::WYSIWYG,
				'rows' => 10,				
				'placeholder' => esc_html__( 'Type your description here', 'rsaddon' ),
			]
		);
		
		$this->end_controls_section();


		$this->start_controls_section(
			'section_heading_style',
			[
				'label' => esc_html__( 'Normal', 'rsaddon' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
            'title_color',
            [
                'label' => esc_html__( 'Title Color', 'rsaddon' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-animated-heading .title-inner .title' => 'color: {{VALUE}};',
                ],                
            ]
        );

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'label' => esc_html__( 'Title Typography', 'rsaddon' ),
				'selector' => '{{WRAPPER}} .rs-animated-heading .title-inner .title',
			]
		);
               
		$this->end_controls_section();


		$this->start_controls_section(
			'animate_heading_style',
			[
				'label' => esc_html__( 'Animated', 'rsaddon' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'animate_style',
			[
				'label'   => esc_html__( 'Animated Style', 'rsaddon' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'clip',
				'options' => [
					'rotate-1'    => esc_html__( 'Rotate', 'rsaddon'),
					'loading-bar' => esc_html__( 'Loading-bar', 'rsaddon'),
					'slide' 	  => esc_html__( 'Slide', 'rsaddon'),
					'slide' 	  => esc_html__( 'Slide', 'rsaddon'),
					'clip' 	  	  => esc_html__( 'Clip', 'rsaddon'),
					'zoom' 	  	  => esc_html__( 'Zoom', 'rsaddon'),
					'push' 	  	  => esc_html__( 'Push', 'rsaddon'),
				],
			]
		);


		$this->add_control(
            'animate_title_color',
            [
                'label' => esc_html__( 'Title Color', 'rsaddon' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-animated-heading .title-inner .title .cd-headline p' => 'color: {{VALUE}};',
                ],                
            ]
        );

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'animate_title_main_typography',
				'label' => esc_html__( 'Main Title Typography', 'rsaddon' ),
				'selector' => '{{WRAPPER}} .rs-animated-heading .title-inner .title .cd-headline',
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'animate_title_typography',
				'label' => esc_html__( 'Title Typography', 'rsaddon' ),
				'selector' => '{{WRAPPER}} .rs-animated-heading .title-inner .title .cd-headline p',
			]
		);

        $this->add_control(
			'highlight_setting',
			[
				'label'   => esc_html__( 'Last Word Color settings', 'rsaddon' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'normal_color',
				'options' => [					
					'normal_color' => esc_html__( 'Normal Color', 'rsaddon'),
					'gradient_color' => esc_html__( 'Gradient Color', 'rsaddon'),
				],
			]
		);

		$this->add_control(
            'animate_last_title_color',
            [
                'label' => esc_html__( 'Title Color', 'rsaddon' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-animated-heading .title-inner .title .cd-headline ' => 'color: {{VALUE}};',
                ], 
                'condition' => [
					'highlight_setting' => 'normal_color',
				],               
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'highlight_color_gradient',
                'label' => esc_html__( 'Gradient Color', 'rsaddon' ),
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .rs-animated-heading .title-inner .title .cd-headline p',
                'condition' => [
					'highlight_setting' => 'gradient_color',
				],
            ]
        );

		$this->add_responsive_control(
			'gap_between_anim',
			[
				'label' => esc_html__( 'Gap Between', 'rsaddon' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'custom' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rs-animated-heading .title-inner .title' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);
            
		$this->end_controls_section();


		$this->start_controls_section(
			'section_heading_style_after',
			[
				'label' => esc_html__( 'After Title', 'rsaddon' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		

		$this->add_control(
            'title_after_style',
            [
                'label' => esc_html__( 'Title Color', 'rsaddon' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-animated-heading .title-inner .title' => 'color: {{VALUE}};',
                ],               
            ]
        );




		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography_after',
				'label' => esc_html__( 'Title Typography', 'rsaddon' ),
				'selector' => '{{WRAPPER}} .rs-animated-heading .title-inner .title',
			]
		); 
		$this->end_controls_section();


		$this->start_controls_section(
			'section_heading_style_des',
			[
				'label' => esc_html__( 'Description Text', 'rsaddon' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
            'des_after_style',
            [
                'label' => esc_html__( 'Description Color', 'rsaddon' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-animated-heading .description p' => 'color: {{VALUE}};',
                ],                
            ]
        );

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'des_typography_after',
				'label' => esc_html__( 'Title Typography', 'rsaddon' ),
				'selector' => '{{WRAPPER}} .rs-animated-heading .description p',
			]
		); 
		$this->add_responsive_control(
		    'des_typography_margin',
		    [
		        'label' => esc_html__( 'Margin', 'rsaddon' ),
		        'type' => Controls_Manager::DIMENSIONS,
		        'size_units' => [ 'px', 'em', '%' ],
		        'selectors' => [
		            '{{WRAPPER}} .rs-animated-heading .description p' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
		        ],
		    ]
		);
		$this->end_controls_section();
	}

	/**
	 * Render rsgallery widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$settings = $this->get_settings_for_display();

		$list_content       = $settings['animated_text'];
		$list_content_text  = esc_html($list_content);
        $list_content_text_lines = explode(",,", $list_content_text);

		$align = $settings['align'] ? $settings['align'] : '';
		    
      ?>
        <div class="rs-animated-heading  <?php echo esc_attr($align);?> <?php echo esc_attr($settings['highlight_setting']);?>">
        	<div class="title-inner">		
				<<?php echo esc_html($settings['title_tag']); ?> class="title">
					<?php echo wp_kses_post($settings['title']); ?>

					<?php if ( !empty($list_content) ) { ?>
						<div class="cd-headline <?php echo esc_attr( $settings['animate_style'] ); ?>">
							<div class="cd-words-wrapper">
								<?php foreach ($list_content_text_lines as $list_content_text_line) {?>
									<p><?php echo $list_content_text_line ?></p>
								<?php } ?>
							</div>
						</div>
					<?php } ?>

					<?php echo wp_kses_post($settings['title_after']); ?>
				</<?php echo esc_html($settings['title_tag']); ?>>
	        </div>
	        <?php if ($settings['content']) { ?>
            	<div class="description">
            		<?php echo wp_kses_post($settings['content']);?>
            	</div>
        	<?php } ?>
        </div>
        <?php 	


        echo '<script>
			jQuery(document).ready(function($) {
				$(".rs-animated-heading .cd-words-wrapper p:first-child").addClass("is-visible");
			});
		</script>';	
	}
}?>