
/* -*- linux-c -*- */

/*
 * Author : Stephen Smalley, <sds@tycho.nsa.gov>
 */

#ifndef _SEPOL_POLICYDB_SERVICES_H_
#define _SEPOL_POLICYDB_SERVICES_H_

/*
 * Security server interface.
 */

#include <sepol/policydb/flask_types.h>
#include <sepol/policydb/policydb.h>
#include <stddef.h>

#ifdef __cplusplus
extern "C" {
#endif

/* Set the policydb and sidtab structures to be used by
   the service functions.  If not set, then these default
   to private structures within libsepol that can only be
   initialized and accessed via the service functions themselves.
   Setting the structures explicitly allows a program to directly
   manipulate them, e.g. checkpolicy populates the structures directly
   from a source policy rather than from a binary policy. */
extern int sepol_set_policydb(policydb_t * p);
extern int sepol_set_sidtab(sidtab_t * s);

/* Modify a policydb for boolean settings. */
int sepol_genbools_policydb(policydb_t * policydb, const char *booleans);

/* Modify a policydb for user settings. */
int sepol_genusers_policydb(policydb_t * policydb, const char *usersdir);

/* Load the security policy. This initializes the policydb
   and sidtab based on the provided binary policy. */
extern int sepol_load_policy(void *data, size_t len);

/*
 * Compute access vectors based on a SID pair for
 * the permissions in a particular class.
 */
extern int sepol_compute_av(sepol_security_id_t ssid,	/* IN */
			    sepol_security_id_t tsid,	/* IN */
			    sepol_security_class_t tclass,	/* IN */
			    sepol_access_vector_t requested,	/* IN */
			    struct sepol_av_decision *avd);	/* OUT */

/* Same as above, but also return the reason(s) for any
   denials of the requested permissions. */
#define SEPOL_COMPUTEAV_TE     0x1U
#define SEPOL_COMPUTEAV_CONS   0x2U
#define SEPOL_COMPUTEAV_RBAC   0x4U
#define SEPOL_COMPUTEAV_BOUNDS 0x8U
extern int sepol_compute_av_reason(sepol_security_id_t ssid,
				   sepol_security_id_t tsid,
				   sepol_security_class_t tclass,
				   sepol_access_vector_t requested,
				   struct sepol_av_decision *avd,
				   unsigned int *reason);

/*
 * Same as above, but also returns the constraint expression calculations
 * whether allowed or denied in a buffer. This buffer is allocated by
 * this call and must be free'd by the caller using free(3). The contraint
 * buffer will contain any constraints in infix notation.
 * If the SHOW_GRANTED flag is set it will show granted and denied
 * constraints. The default is to show only denied constraints.
 */
#define SHOW_GRANTED 1
extern int sepol_compute_av_reason_buffer(sepol_security_id_t ssid,
				   sepol_security_id_t tsid,
				   sepol_security_class_t tclass,
				   sepol_access_vector_t requested,
				   struct sepol_av_decision *avd,
				   unsigned int *reason,
				   char **reason_buf,
				   unsigned int flags);

/*
 * Returns the mls/validatetrans constraint expression calculations in
 * a buffer that must be free'd by the caller using free(3).
 * If the SHOW_GRANTED flag is set it will show granted and denied
 * mls/validatetrans (the default is to show only those denied).
 */
extern int sepol_validate_transition_reason_buffer(sepol_security_id_t oldsid,
					sepol_security_id_t newsid,
					sepol_security_id_t tasksid,
					sepol_security_class_t tclass,
					char **reason_buf,
					unsigned int flags);

/*
 * Return a class ID associated with the class string representation
 * specified by `class_name'.
 */
extern int sepol_string_to_security_class(const char *class_name,
					sepol_security_class_t  *tclass);

/*
 * Return a permission av bit associated with tclass and the string
 * representation of the `perm_name'.
 */
extern int sepol_string_to_av_perm(sepol_security_class_t tclass,
					const char *perm_name,
					sepol_access_vector_t *av);

/*
 * Compute a SID to use for labeling a new object in the 
 * class `tclass' based on a SID pair.  
 */
extern int sepol_transition_sid(sepol_security_id_t ssid,	/* IN */
				sepol_security_id_t tsid,	/* IN */
				sepol_security_class_t tclass,	/* IN */
				sepol_security_id_t * out_sid);	/* OUT */

/*
 * Compute a SID to use when selecting a member of a 
 * polyinstantiated object of class `tclass' based on 
 * a SID pair.
 */
extern int sepol_member_sid(sepol_security_id_t ssid,	/* IN */
			    sepol_security_id_t tsid,	/* IN */
			    sepol_security_class_t tclass,	/* IN */
			    sepol_security_id_t * out_sid);	/* OUT */

/*
 * Compute a SID to use for relabeling an object in the 
 * class `tclass' based on a SID pair.  
 */
extern int sepol_change_sid(sepol_security_id_t ssid,	/* IN */
			    sepol_security_id_t tsid,	/* IN */
			    sepol_security_class_t tclass,	/* IN */
			    sepol_security_id_t * out_sid);	/* OUT */

/*
 * Write the security context string representation of 
 * the context associated with `sid' into a dynamically
 * allocated string of the correct size.  Set `*scontext'
 * to point to this string and set `*scontext_len' to
 * the length of the string.
 */
extern int sepol_sid_to_context(sepol_security_id_t sid,	/* IN */
				sepol_security_context_t * scontext,	/* OUT */
				size_t * scontext_len);	/* OUT */

/*
 * Return a SID associated with the security context that
 * has the string representation specified by `scontext'.
 */
extern int sepol_context_to_sid(const sepol_security_context_t scontext,	/* IN */
				size_t scontext_len,	/* IN */
				sepol_security_id_t * out_sid);	/* OUT */

/*
 * Generate the set of SIDs for legal security contexts
 * for a given user that can be reached by `fromsid'.
 * Set `*sids' to point to a dynamically allocated 
 * array containing the set of SIDs.  Set `*nel' to the
 * number of elements in the array.
 */
extern int sepol_get_user_sids(sepol_security_id_t callsid,
			       char *username,
			       sepol_security_id_t ** sids, uint32_t * nel);

/*
 * Return PK     =O[(  (    sepol/policydb/hashtab.hnu [        PK     =O[B:      sepol/policydb/link.hnu [        PK     =O[ ?ch  h    sepol/policydb/ebitmap.hnu [        PK     =O[c      sepol/policydb/avtab.hnu [        PK     =O[sy>J      sepol/policydb/polcaps.hnu [        PK     =O[\w
  
    sepol/policydb/context.hnu [        PK     =O[      sepol/policydb/mls_types.hnu [        PK     =O[z      sepol/policydb/flask.hnu [        PK     =O[W      sepol/policydb/module.hnu [        PK     =O[rΊ}  }    sepol/policydb/conditional.hnu [        PK     =O[MH  H    sepol/policydb/expand.hnu [        PK     =O[a(      sepol/policydb/sidtab.hnu [        PK     =O[V$N  N    sepol/policydb/symtab.hnu [        PK     =O[6çj^  ^  
  sepol/sepol.hnu [        PK     =O[Ex
  x
    sepol/module.hnu [        PK     =O[#q      
  sepol/nodes.hnu [        PK     =O[OFBI  I    sepol/module_to_cil.hnu [        PK     =O[Ís      sepol/ibpkey_record.hnu [        PK     =O[W5      sepol/boolean_record.hnu [        PK     =O[~T      sepol/port_record.hnu [        PK     =O[F:P	  P	    sepol/user_record.hnu [        PK     =O[W    
  sepol/users.hnu [        PK     =O[      pcre2.hnu [        PK     =O[凁<
  
    idna.hnu [        PK     =O[4      error.hnu [        PK     =O[/R;  ;    lber.hnu [        PK     =O[A{  {    pcre.hnu [        PK     =O[pR  pR    sound/asequencer.hnu [        PK     =O[et5+  +    sound/firewire.hnu [        PK     =O[A
  
    sound/sb16_csp.hnu [        PK     =O[      sound/usb_stream.hnu [        PK     =O[N;+      sound/tlv.hnu [        PK     =O[NLG      sound/asound.hnu [        PK     =O[Ē8W  W    sound/asoc.hnu [        PK     =O[y  y    sound/sof/header.hnu [        PK     =O[bN\  \    sound/sof/abi.hnu [        PK     =O[gٙ      sound/sof/tokens.hnu [        PK     =O[_~La      sound/sof/fw.hnu [        PK     =O[qR&  &    sound/compress_offload.hnu [        PK     =O[nI  I    sound/emu10k1.hnu [        PK     =O[)gc    
  sound/hdspm.hnu [        PK     =O[CM@  M@    sound/compre