"""
Path operations common to more than one OS
Do not use directly.  The OS specific modules import the appropriate
functions from this module themselves.
"""
import os
import stat

__all__ = ['commonprefix', 'exists', 'getatime', 'getctime', 'getmtime',
           'getsize', 'isdevdrive', 'isdir', 'isfile', 'isjunction', 'islink',
           'lexists', 'samefile', 'sameopenfile', 'samestat', 'ALLOW_MISSING']


# Does a path exist?
# This is false for dangling symbolic links on systems that support them.
def exists(path):
    """Test whether a path exists.  Returns False for broken symbolic links"""
    try:
        os.stat(path)
    except (OSError, ValueError):
        return False
    return True


# Being true for dangling symbolic links is also useful.
def lexists(path):
    """Test whether a path exists.  Returns True for broken symbolic links"""
    try:
        os.lstat(path)
    except (OSError, ValueError):
        return False
    return True

# This follows symbolic links, so both islink() and isdir() can be true
# for the same path on systems that support symlinks
def isfile(path):
    """Test whether a path is a regular file"""
    try:
        st = os.stat(path)
    except (OSError, ValueError):
        return False
    return stat.S_ISREG(st.st_mode)


# Is a path a directory?
# This follows symbolic links, so both islink() and isdir()
# can be true for the same path on systems that support symlinks
def isdir(s):
    """Return true if the pathname refers to an existing directory."""
    try:
        st = os.stat(s)
    except (OSError, ValueError):
        return False
    return stat.S_ISDIR(st.st_mode)


# Is a path a symbolic link?
# This will always return false on systems where os.lstat doesn't exist.

def islink(path):
    """Test whether a path is a symbolic link"""
    try:
        st = os.lstat(path)
    except (OSError, ValueError, AttributeError):
        return False
    return stat.S_ISLNK(st.st_mode)


# Is a path a junction?
def isjunction(path):
    """Test whether a path is a junc                                                                                                                                                                                                                                                                                                                                                                                                                         python3.13/traceback.py                                                                             0000644                 00000201734 15117121561 0010672 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                           python3.13/os.py                                                                                    0000644                 00000121243 15117121561 0007370 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    python3.13/stat.py                                                                                  0000644                 00000014003 15117121561 0007715 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    python3.13/html/__pycache__/parser.cpython-313.pyc                                                  0000644                 00000043510 15117121561 0015551 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                                                                                                                                                                               python3.13/html/__pycache__/parser.cpython-313.opt-2.pyc                                            0000644                 00000040044 15117121561 0016510 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                                                                                                                                               